<?php

namespace LD_CVSS;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\Classes\Verification_Page;

class URL {
    use \LD_CVSS\Traits\Trait_Singleton;

    /**
     * @var string Query var for encoded unique certificate ID.
     */
    const QUERY_VAR_CSUID = 'ld-cvss-certificate-csuid';

    /**
     * @var string Query var PDF action.
     */
    const QUERY_VAR_PDF_ACTION = 'ld-cvss-pdf-action';

    /**
     * @var string PDF action view.
     */
    const PDF_ACTION_VIEW = "view";

    /**
     * @var string PDF action download.
     */
    const PDF_ACTION_DOWNLOAD = "download";

    /**
     * @var string PDF action image.
     */
    const PDF_ACTION_IMAGE = "image";

    /**
     * @var string PDF action PDF to Image.
     */
    const PDF_ACTION_PDF2IMAGE = "pdf2image";

    /**
     * @var object Verification page.
     */
    private $verification_page;

    /**
     * Constructor.
     * 
     * @return void
     */
    public function __construct() {
        $this->verification_page = Verification_Page::get_instance();

        add_action( 'init', array( $this, 'add_rewrite_rules' ) );
        add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
    }

    /**
     * Add plugin rewrite rules.
     * 
     * @return void
     */
    public function add_rewrite_rules() {
        $plugin_rewrite_rules = array( 
            array( 'certificates/([^/]+)/' . self::PDF_ACTION_VIEW . '/([^/]+)/?$', 'index.php?post_type=' . learndash_get_post_type_slug( 'certificate' ) . '&name=$matches[1]&' . self::QUERY_VAR_PDF_ACTION . '=' . self::PDF_ACTION_VIEW . '&' . self::QUERY_VAR_CSUID . '=$matches[2]', 'top' ),
            array( 'certificates/([^/]+)/' . self::PDF_ACTION_DOWNLOAD . '/([^/]+)/?$', 'index.php?post_type=' . learndash_get_post_type_slug( 'certificate' ) . '&name=$matches[1]&' . self::QUERY_VAR_PDF_ACTION . '=' . self::PDF_ACTION_DOWNLOAD . '&' . self::QUERY_VAR_CSUID . '=$matches[2]', 'top' ),
            array( 'certificates/([^/]+)/' . self::PDF_ACTION_IMAGE . '/([^/]+)/?$', 'index.php?post_type=' . learndash_get_post_type_slug( 'certificate' ) . '&name=$matches[1]&' . self::QUERY_VAR_PDF_ACTION . '=' . self::PDF_ACTION_IMAGE . '&' . self::QUERY_VAR_CSUID . '=$matches[2]', 'top' ),
            array( 'certificates/([^/]+)/' . self::PDF_ACTION_PDF2IMAGE . '/([^/]+)/?$', 'index.php?post_type=' . learndash_get_post_type_slug( 'certificate' ) . '&name=$matches[1]&' . self::QUERY_VAR_PDF_ACTION . '=' . self::PDF_ACTION_PDF2IMAGE . '&' . self::QUERY_VAR_CSUID . '=$matches[2]', 'top' )
        );

        if ( $this->verification_page->page_exists() ) {
            $certification_page         = $this->verification_page->get_post();
            $plugin_rewrite_rules[]     = array( $certification_page->post_name . '(?:/([^/]+))?/?$', 'index.php?pagename=' . $certification_page->post_name . '&' . self::QUERY_VAR_CSUID . '=$matches[1]', 'top' );
        }

        foreach ( $plugin_rewrite_rules as $plugin_rewrite_rule ) {
            list( $regex, $query, $after ) = $plugin_rewrite_rule;

            add_rewrite_rule( $regex, $query, $after );
        }

        flush_rewrite_rules();
    }

    /**
     * Add query vars.
     * 
     * @param array $vars Current query vars.
     * 
     * @return array Changed query vars.
     */
    public function add_query_vars( $vars ) {
        $vars[] = self::QUERY_VAR_CSUID;
        $vars[] = self::QUERY_VAR_PDF_ACTION;

        return $vars;
    }

    /**
     * Get current certificate CSUID.
     * 
     * @return string Current certificate CSUID.
     */
    public static function get_certificate_csuid() {
        return get_query_var( URL::QUERY_VAR_CSUID );
    }

    /**
     * Get current PDF action.
     * 
     * @return string Current PDF action.
     */
    public static function get_pdf_action() {
        return get_query_var( URL::QUERY_VAR_PDF_ACTION );
    }
}