<?php

namespace LD_CVSS\Metaboxes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

/**
 * Used for plugin upgrades.
 */
class Certificate_Description {
    /**
     * @var string Meta key description.
     */
    const META_KEY_DESCRIPTION = '_ld_cvss_certificate_description';

    /**
     * @var string Meta key migrated.
     */
    const META_KEY_MIGRATED = 'ld_cvss_migrated_from_excerpt_to_pm';

    /**
     * @var string Field certificate description.
     */
    const FIELD = 'ld_cvss_certificate_description';

    /**
     * @var string Nonce name.
     */
    const NONCE_NAME = 'ld_cvss_nonce_action_certificate_description';

    /**
     * @var string Nonce action.
     */
    const NONCE_ACTION = 'ld_cvss_nonce_name_certificate_description';

    use \LD_CVSS\Traits\Trait_Singleton;

    /**
     * Constructor.
     * 
     * @return void
     */
    public function __construct() {
        add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
        add_action( 'save_post', array( $this, 'save_meta_box' ) );
    }

    /**
     * Add meta box.
     * 
     * @return void
     */
    public function add_meta_box() {
        add_meta_box(
            'ld_cvss_meta_box_certificate_description',
            __( 'Certificate Description', 'LD-CVSS' ),
            array( $this, 'render_meta_box' ),
            learndash_get_post_type_slug( 'certificate' ),
            'normal',
            'high'
        );
    }

    /**
     * Render meta box.
     * 
     * @return void
     */
    public function render_meta_box( $post ) {
        $description = '';

        if ( 
            ! empty( get_the_excerpt( $post->ID ) ) &&
            get_post_meta( $post->ID, self::META_KEY_MIGRATED , true ) != 'yes'
        ) {
            $description = get_the_excerpt( $post->ID );
        } else {
            $description = get_post_meta( $post->ID, self::META_KEY_DESCRIPTION, true );
        }

        wp_editor( $description, self::FIELD, array(
            'wpautop'             => ! has_blocks(),
            'media_buttons'       => true,
            'default_editor'      => 'html',
            'drag_drop_upload'    => true,
            'textarea_name'       => self::FIELD,
            'textarea_rows'       => 20,
            'tabindex'            => '',
            'tabfocus_elements'   => ':prev,:next',
            'editor_css'          => '',
            'editor_class'        => '',
            'teeny'               => true,
            '_content_editor_dfw' => true,
            'tinymce'             => true,
            'quicktags'           => true,
        ) );

        wp_nonce_field( self::NONCE_ACTION, self::NONCE_NAME );
    }

    /**
     * Save meta box.
     * 
     * @return void
     */
    public function save_meta_box( $post_id ) {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return $post_id;
        }

        if ( get_post_type( $post_id ) != learndash_get_post_type_slug( 'certificate' ) ) {
            return $post_id;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return $post_id;
        }

        $nonce_name = ! empty( $_POST[ self::NONCE_NAME ] ) ? $_POST[ self::NONCE_NAME ] : '';

        if ( ! wp_verify_nonce( $nonce_name, self::NONCE_ACTION ) ) {
            return $post_id;
        }
        
        $description = $_POST[ self::FIELD ];

        if ( 
            ! empty( get_the_excerpt( $post_id ) ) &&
            get_post_meta( $post_id, self::META_KEY_MIGRATED , true ) != 'yes'
        ) {
            $description = get_the_excerpt( $post_id );
        }

        update_post_meta( $post_id, self::META_KEY_MIGRATED, 'yes' );
        update_post_meta( $post_id, self::META_KEY_DESCRIPTION, $description );
    }
}