<?php

namespace LD_CVSS;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\Classes\Options;
use LD_CVSS\Classes\Template;
use LD_CVSS\Classes\Helper;
use LD_CVSS\Classes\Verification_Page;
use LD_CVSS\Classes\Certificate;
use LD_CVSS\Classes\Social_Buttons;
use LD_CVSS\Classes\Certificate_Access;
use LD_CVSS\Deprecated\Base64_Security;

class Frontend {
    use \LD_CVSS\Traits\Trait_Singleton;

    /**
     * @var string PDF image mime type.
     */
    const PDF_IMAGE_MIME_TYPE = 'image/jpeg';

    /**
     * @var string Shortcode to display verification page.
     */
    const SHORTCODE_VERIFICATION_PAGE = 'certificate_verification';
    
    /**
     * @var string Shortcode to display unique certificate ID (CSUID).
     */
    const SHORTCODE_CERTIFICATE_ID = 'ld_cvss_certificate_id';

    /**
     * @var string Shortcode to display QR Code.
     */
    const SHORTCODE_CERTIFICATE_QR_CODE = 'ld_cvss_certificate_qr_code';

    /**
     * @var array Verificateion page templates.
     */
    const PAGE_TEMPLATES = array(
        'template-01',
        'template-02'
    );

    /**
     * @var string AJAX action to convert image from PDF.
     */
    const AJAX_ACTION_PDF2IMAGE = 'ld_cvss_pdf2image';

    /**
     * @var string AJAX action to handle search form.
     */
    const AJAX_ACTION_SEARCH_FORM = 'ld_cvss_search_form';

    /**
     * @var object Plugin options.
     */
    private $options;

    /**
     * @var object Verification page.
     */
    private $verification_page;

    /**
     * @var object Certificate.
     */
    private $certificate;

    /**
     * Constructor.
     * 
     * @return void
     */
    public function __construct() {
        Certificate_Hooks::get_instance();

        $this->options              = Options::get_instance();
        $this->verification_page    = Verification_Page::get_instance();

        add_action( 'template_redirect', array( $this, 'template_redirect' ), 1 );
        add_action( 'wp', array( $this, 'wp' ) );
        add_action( 'body_class', array( $this, 'body_class' ), 99 );
        add_action( 'wp_enqueue_scripts', array( $this, 'frontend_assets' ) );
        add_action( 'wp_head', array( $this, 'opengraph_meta_tags' ), 1 );
        add_shortcode( self::SHORTCODE_VERIFICATION_PAGE, array( $this, 'do_shortcode_verification_page' ) );
        add_shortcode( self::SHORTCODE_CERTIFICATE_ID, array( $this, 'do_shortcode_certificate_id' ) );
        add_shortcode( self::SHORTCODE_CERTIFICATE_QR_CODE, array( $this, 'do_shortcode_certificate_qr_code' ) );
        add_action( 'learndash_certification_content_write_cell_after', array( $this, 'ld_download_pdf' ), 10, 2 );
        add_filter( 'learndash_certificate_builder_pdf_output_mode', array( $this, 'ld_certificate_builder_download_pdf' ) );
        add_filter( 'learndash_certificate_builder_pdf_name', array( $this, 'ld_certificate_builder_download_file_name' ) );
        add_action( 'wp_ajax_' . self::AJAX_ACTION_PDF2IMAGE, array( $this, 'ajax_pdf2image' ) );
        add_action( 'wp_ajax_nopriv_' . self::AJAX_ACTION_PDF2IMAGE, array( $this, 'ajax_pdf2image' ) );
        add_action( 'wp_ajax_' . self::AJAX_ACTION_SEARCH_FORM, array( $this, 'ajax_search_form' ) );
        add_action( 'wp_ajax_nopriv_' . self::AJAX_ACTION_SEARCH_FORM,  array( $this, 'ajax_search_form' ) );
        /* Hide default certificate purchase buttons generated by LearnDash Pay for Quiz & Certificate. */
        add_filter( 'ld_pfqc_display_certificate_purchase_button', '__return_false' );
    }

    /**
     * Redirect to the WC checkout page if the Add to cart button is clicked.
     *
     * @return void
     */
	public function template_redirect() {
        if ( ( $this->verification_page->is_verification_page() || learndash_get_post_type_slug( 'quiz' ) == get_post_type() ) && ! empty( $_GET['add-to-cart'] ) ) {
           if ( function_exists( 'wc_get_checkout_url' ) ) {
               wp_redirect( wc_get_checkout_url() );
               exit;
           }
        }
    }

    /**
     * WP ready.
     *
     * @return void
     */
    public function wp() {
        global $wp_query;

        $certificate_csuid  = Helper::decode_csuid( URL::get_certificate_csuid() );
        $certificate        = new Certificate( $certificate_csuid['cert_id'], $certificate_csuid['source_id'], $certificate_csuid['user_id'] );

        if ( $certificate->is_valid() ) {
            $this->certificate = $certificate;
        }

        if ( $this->verification_page->is_verification_page() ) {
            /** 
             * Redirect to the new URL structure of the verification page 
             * if old $_GET['certificate-key'] exists and checked. 
             * */
            if ( ! empty( $_GET['certificate-key'] ) ) {
                $decoded_string = Base64_Security::decode( $_GET['certificate-key'] );

                if ( $decoded_string ) {
                    $decoded_data = Base64_Security::get_decoded_data( $decoded_string );

                    if ( $decoded_data ) {
                        $cert_id = ( int ) learndash_get_setting( $decoded_data['source_id'], 'certificate' );

                        if ( $cert_id ) {
                            $certificate = new Certificate( $cert_id, $decoded_data['source_id'], $decoded_data['user_id'] );

                            if ( $certificate->is_valid() ) {
                                wp_redirect( $certificate->get_public_url(), 301 );
                                exit;
                            }
                        }
                    }
                }
            }
        } 

        if ( URL::get_pdf_action() && $this->certificate ) {
            $current_user   = wp_get_current_user();
            $get_args       = $this->certificate->get_source()->get_pdf_url_args();

            foreach ( $get_args as $get_arg_name => $get_arg_value ) {
                /* Set $_REQUEST params for correct PDF processing */
                $_REQUEST[ $get_arg_name ] = $get_arg_value;
                /* Set $_GET params for correct PDF processing */
                $_GET[ $get_arg_name ] = $get_arg_value;
            }

            if ( isset( $get_args['user_id'] ) ) {
                /* Temporarily set a certificate user as the current user. Needed for shortcodes to work properly. */
                wp_set_current_user( $get_args['user_id'] );
            }

            if ( Certificate_Access::is_purchased( $this->certificate ) ) {
                switch ( URL::get_pdf_action() ) {
                    case URL::PDF_ACTION_VIEW:
                        $this->pdf_output( $this->certificate->get_id(), $this->certificate->get_source_id(), $this->certificate->get_user_id() );
                        break;
                    case URL::PDF_ACTION_DOWNLOAD:
                        $this->pdf_output( $this->certificate->get_id(), $this->certificate->get_source_id(), $this->certificate->get_user_id() );
                        break;
                    case URL::PDF_ACTION_IMAGE:
                        $this->pdf_image_output();
                        break;
                    case URL::PDF_ACTION_PDF2IMAGE:
                        $this->render_pdf2image_iframe_content();
                        break;
                }
            } elseif ( URL::get_pdf_action() == URL::PDF_ACTION_VIEW ) {
                $this->pdf_output( $this->certificate->get_id(), $this->certificate->get_source_id(), $this->certificate->get_user_id() );
            } else {
                $wp_query->set_404();
                status_header( 404 );
                get_template_part( 404 );
            }

            wp_set_current_user( $current_user->ID );

            exit;
        }

        if ( 
            URL::get_certificate_csuid() && ! $this->certificate || 
            $this->verification_page->is_verification_page() && $this->certificate && ! Certificate_Access::is_purchased( $this->certificate ) && 
            ( 
                ! Certificate_Access::is_owner( $this->certificate ) && 
                ! Certificate_Access::is_viewable( $this->certificate ) 
            )
        ) {
            $wp_query->set_404();
            status_header( 404 );
            get_template_part( 404 );
            exit;
        }
    }

    /**
     * Add custom css class to body.
     *
     * @param array $classes Current classes.
     * 
     * @return array Changed classes.
     */
    public function body_class( $classes ) {
        if ( $this->verification_page->is_verification_page() ) {
            $template = 'ld-cvss-' . $this->options->get( 'verification_page_template', self::PAGE_TEMPLATES[0] );

            array_unshift( $classes, $template );
        }

        return $classes;
    }

    /**
     * Enqueue frontend assets.
     * 
     * @return void
     */
    public function frontend_assets() {
        // wp_enqueue_style( 'ld_cvss-font-awesome', LD_CVSS_URL_ASSETS_CSS . '/font-awesome' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_enqueue_style( 'ld_cvss-social-buttons', LD_CVSS_URL_ASSETS_CSS . '/social-buttons' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_enqueue_style( 'ld_cvss-social-buttons', LD_CVSS_URL_ASSETS_CSS . '/social-buttons-rtl' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_style_add_data( 'ld_cvss-social-buttons', 'rtl', 'replace' );

        if ( ! $this->verification_page->is_verification_page() ) {
            return;
        }

        wp_enqueue_style( 'ld_cvss-nprogress', LD_CVSS_URL_ASSETS_CSS . '/nprogress' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_enqueue_style( 'ld_cvss-nprogress', LD_CVSS_URL_ASSETS_CSS . '/nprogress-rtl' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_style_add_data( 'ld_cvss-nprogress', 'rtl', 'replace' );
        wp_enqueue_style( 'ld_cvss-frontend', LD_CVSS_URL_ASSETS_CSS . '/frontend' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_enqueue_style( 'ld_cvss-frontend', LD_CVSS_URL_ASSETS_CSS . '/frontend-rtl' . LD_CVSS_ASSETS_SUFFIX . '.css', array(), LD_CVSS_VERSION );
        wp_style_add_data( 'ld_cvss-frontend', 'rtl', 'replace' );

        $dependencies = array( 'jquery', 'ld_cvss-nprogress' );

        if ( $this->certificate ) {
            wp_enqueue_script( 'ld_cvss-pdfjs', LD_CVSS_URL_ASSETS_JS . '/pdf' . LD_CVSS_ASSETS_SUFFIX . '.js', array(), true, LD_CVSS_VERSION );
            wp_enqueue_script( 'ld_cvss-pdf2image', LD_CVSS_URL_ASSETS_JS . '/pdf2image' . LD_CVSS_ASSETS_SUFFIX . '.js', array( 'jquery', 'ld_cvss-pdfjs' ), true, LD_CVSS_VERSION );
            wp_localize_script( 'ld_cvss-pdf2image', 'ld_cvss_pdf2image_data', array(
                'worker'            => LD_CVSS_URL_ASSETS_JS . '/pdf-worker' . LD_CVSS_ASSETS_SUFFIX . '.js',
                'pdf'               => $this->certificate->get_pdf_view_url(),
                'is_purchased'      => Certificate_Access::is_purchased( $this->certificate ) ? 'true' : 'false'
            ) );

            $dependencies[] = 'ld_cvss-pdf2image';
        }
        
        wp_enqueue_script( 'ld_cvss-nprogress', LD_CVSS_URL_ASSETS_JS . '/nprogress' . LD_CVSS_ASSETS_SUFFIX . '.js', array(), true, LD_CVSS_VERSION );
        wp_enqueue_script( 'ld_cvss-frontend', LD_CVSS_URL_ASSETS_JS . '/frontend' . LD_CVSS_ASSETS_SUFFIX . '.js', $dependencies, true, LD_CVSS_VERSION );
        wp_localize_script( 'ld_cvss-frontend', 'ld_cvss_search_form', array(
            'ajax_url'      => admin_url( 'admin-ajax.php' ),
            'ajax_action'   => self::AJAX_ACTION_SEARCH_FORM,
            'ajax_nonce'    => wp_create_nonce( self::AJAX_ACTION_SEARCH_FORM )
        ) );
    }

    /**
     * Display Opengraph meta tags.
     *
     * @return void
     */
    public function opengraph_meta_tags() {
        if ( ! $this->verification_page->is_verification_page() ) {
            return;
        }

        if ( ! $this->certificate ) {
            return;
        }

        if ( ! Certificate_Access::is_purchased( $this->certificate ) ) {
            return;
        }

        if ( ! apply_filters( 'ld_cvss_add_opengraph_tags', true ) ) {
            return;
        }

        remove_action( 'wp_head', 'rel_canonical' );

        $opengraph_general_tags = array(
            'og:locale'         => get_locale(),
            'og:type'           => 'article',
            'og:title'          => sanitize_text_field( get_the_title( $this->certificate->get_id() ) ),
            'og:description'    => __( 'Click here to view the certificate details', 'LD-CVSS' ),
            'og:url'            => esc_url( $this->verification_page->get_current_url() ),
            'og:site_name'      => get_bloginfo( 'name' ),
            'og:updated_time'   => get_the_date( DATE_W3C, $this->certificate->get_id() )
        );

        $opengraph_twitter_tags = array(
            'twitter:card'          => 'summary_large_image',
            'twitter:title'         => sanitize_text_field( get_the_title( $this->certificate->get_id() ) ),
            'twitter:description'   => __( 'Click here to view the certificate details', 'LD-CVSS' ),
        );

        $opengraph_general_image_tags  = array();
        $opengraph_twitter_image_tags  = array();

        if ( $this->options->get( 'social_sharing_with_image', true ) ) {
            $pdf_image_url = $this->certificate->get_pdf_image_url();
            $image_editor = wp_get_image_editor( $this->certificate->get_pdf_image_md5_path() );

            if ( $pdf_image_url && ! is_wp_error( $image_editor ) ) {
                $pdf_image_size = $image_editor->get_size();

                $opengraph_general_image_tags = array(
                    'og:image'              => $pdf_image_url,
                    'og:image:url'          => $pdf_image_url,
                    'og:image:secure'       => $pdf_image_url,
                    'og:image:secure_url'   => $pdf_image_url,
                    'og:image:width'        => $pdf_image_size['width'],
                    'og:image:height'       => $pdf_image_size['height'],
                    'og:image:alt'          => sanitize_text_field( get_the_title( $this->certificate->get_id() ) ),
                    'og:image:type'         => self::PDF_IMAGE_MIME_TYPE
                );

                $opengraph_twitter_image_tags = array(
                    'twitter:image' => $pdf_image_url
                );
            }
        }

        $opengraph_general_tags            = apply_filters( 'ld_cvss_opengraph_general_tags', $opengraph_general_tags, $this->certificate );
        $opengraph_general_image_tags      = apply_filters( 'ld_cvss_opengraph_general_image_tags', $opengraph_general_image_tags, $this->certificate );
        $opengraph_twitter_tags            = apply_filters( 'ld_cvss_opengraph_twitter_tags', $opengraph_twitter_tags, $this->certificate );
        $opengraph_twitter_image_tags      = apply_filters( 'ld_cvss_opengraph_twitter_image_tags', $opengraph_twitter_image_tags, $this->certificate );

        $opengraph_tags = array_merge( $opengraph_general_tags, $opengraph_general_image_tags, $opengraph_twitter_tags, $opengraph_twitter_image_tags );

        printf( '<link rel="canonical" href="%s" />', esc_url( $this->verification_page->get_current_url() ) );

        foreach ( $opengraph_tags as $opengraph_tag_name => $opengraph_tag_content ) {
            printf( '<meta property="%s" content="%s" />', $opengraph_tag_name, $opengraph_tag_content );
        }
    }

    /**
     * Do shortcode to display Verification page.
     * 
     * @return string HTML content of the shortcode.
     */
    public function do_shortcode_verification_page() {
        if ( $this->verification_page->is_verification_page() ) {
            if ( $this->certificate ) {
                $page_template          = $this->options->get( 'verification_page_template', self::PAGE_TEMPLATES[0] );
                $page_template_data     = apply_filters( 'ld_cvss_verification_page_data', 
                    array(
                        'certificate'       => array(
                            'post'              => $this->certificate->get_post(),
                            'csuid'             => $this->certificate->get_csuid(),
                            'title'             => $this->certificate->get_title(),
                            'description'       => $this->certificate->get_description(),
                            'pdf_meta'          => $this->certificate->get_pdf_meta(),
                            'type'              => $this->certificate->get_type(),
                            'source'            => $this->certificate->get_source(),
                            'course'            => $this->certificate->get_course(),
                            'issue_date'        => date_i18n( apply_filters( 'ld_cvss_certificate_issue_date_format', $this->options->get( 'certificate_issue_date_format', get_option( 'date_format' ) ) ), $this->certificate->get_issue_date() ),
                            'instructor'        => $this->certificate->get_instructor(),
                            'recipient'         => $this->certificate->get_recipient(),
                            'pdf_view_url'      => $this->certificate->get_pdf_view_url(),
                            'pdf_download_url'  => $this->certificate->get_pdf_download_url(),
                        ),
                        'social_buttons'    => Social_Buttons::get( $this->certificate )
                    ),
                    $this->certificate
                );

                add_filter( 'ld_pfqc_certificate_purchase_button_title', array( $this, 'ld_pfqc_certificate_purchase_button_title' ) );
                add_filter( 'ld_pfqc_wc_certificate_purchase_button_title_variation', array( $this, 'ld_pfqc_wc_certificate_purchase_button_title_variation' ), 10, 2 );

                $page_template_data['ld_pay_for_quiz_certificate'] = array(
                    'is_owner'          => Certificate_Access::is_owner( $this->certificate ),
                    'is_viewable'       => Certificate_Access::is_viewable( $this->certificate ),
                    'is_purchased'      => Certificate_Access::is_purchased( $this->certificate ),
                    'purchase_button'   => Certificate_Access::get_purchase_button( $this->certificate )
                );

                remove_filter( 'ld_pfqc_certificate_purchase_button_title', array( $this, 'ld_pfqc_certificate_purchase_button_title' ) );
                remove_filter( 'ld_pfqc_wc_certificate_purchase_button_title_variation', array( $this, 'ld_pfqc_wc_certificate_purchase_button_title_variation' ), 10, 2 );

                return Template::get( "{$page_template}.php", $page_template_data );
            } else {
                return Template::get( 'search-form.php' );
            }
        }

        return '';
    }

    /**
     * Change certificate purchase button title generated by LearnDash Pay for Quiz & Certificate.
     *
     * @param string $title Default title.
     * 
     * @return string Changed title.
     */
    public function ld_pfqc_certificate_purchase_button_title( $title ) {
        return apply_filters( 'ld_cvss_ld_pfqc_certificate_purchase_button_title', __( 'Purchase PDF', 'LD-CVSS' ) );
    }

    /**
     * Change WC certificate purchase button title variation generated by LearnDash Pay for Quiz & Certificate.
     *
     * @param string $button_source Button source.
     * @param string $source_type Source type.
     * 
     * @return string Changed button source.
     */
    public function ld_pfqc_wc_certificate_purchase_button_title_variation( $button_source, $source_type ) {
        return apply_filters( 'ld_cvss_ld_pfqc_certificate_purchase_button_title_variation', 'button_title' );
    }

    /**
     * Do shortcode to display unique certificate ID (CSUID).
     * 
     * @return string HTML content of the shortcode.
     */
    public function do_shortcode_certificate_id() {
        $certificate = null;

        if ( URL::get_certificate_csuid() ) {
            $certificate_csuid  = Helper::decode_csuid( URL::get_certificate_csuid() );
            $certificate        = new Certificate( $certificate_csuid['cert_id'], $certificate_csuid['source_id'], $certificate_csuid['user_id'] );
        } elseif (
            get_post_type() == learndash_get_post_type_slug( 'certificate' ) && 
            ( ! empty( $_GET[ 'course_id' ] ) || ! empty( $_GET[ 'quiz' ] ) || ! empty( $_GET[ 'group_id' ] ) )
        ) {

            if ( ! empty( $_GET[ 'course_id' ] ) ) {
                $source_id = absint( $_GET[ 'course_id' ] );
            } elseif ( ! empty( $_GET[ 'quiz' ] ) ) {
                $source_id = absint( $_GET[ 'quiz' ] );
            } elseif ( ! empty( $_GET[ 'group_id' ] ) ) {
                $source_id = absint( $_GET[ 'group_id' ] );
            }

            if ( $source_id ) {
                $cert_id        = get_the_ID();
                $user_id        = ! empty( $_GET['user_id'] ) ? absint( $_GET['user_id'] ) : get_current_user_id();
                $certificate    = new Certificate( $cert_id, $source_id, $user_id );
            }
        }

        if ( $certificate && $certificate->is_valid() ) {
            return $certificate->get_csuid();
        }

        return '';
    }

    public function do_shortcode_certificate_qr_code( $atts ) {
        $certificate_csuid  = Helper::decode_csuid( URL::get_certificate_csuid() );
        $certificate        = new Certificate( $certificate_csuid['cert_id'], $certificate_csuid['source_id'], $certificate_csuid['user_id'] );

        if ( ! $certificate->is_valid() ) {
            return '';
        }

        $atts = apply_filters( 'ld_cvss_certificate_qr_code_atts',
            shortcode_atts( array(
                'size'      => '100',
                'inline'    => 'true',
                'align'     => 'left',
                'url'       => $certificate->get_public_url()
            ), $atts ),
            $certificate
        );

        $qrcode_url         = sprintf( 'https://chart.googleapis.com/chart?cht=qr&chs=%1$sx%1$s&chl=%2$s', $atts['size'], $atts['url'] );
        $qrcode_image       = sprintf( '<img src="%1$s" width="%2$s" width="%2$s">', $qrcode_url, $atts['size'] );
        $qrcode             = sprintf( '<span style="margin: 0; line-height: 0;">%s</span>', $qrcode_image );

        if ( $atts['inline'] != 'true' ) {
            $qrcode = sprintf( '<div style="margin: 0; line-height: 0; text-align:%s;">%s</div>', $atts['align'], $qrcode_image );
        }

        return $qrcode;
    }

    /**
     * Output PDF
     *
     * @param int $cert_id Certificate ID.
     * @param int $source_id Source ID.
     * @param int $user_id User ID.
     * 
     * @return void
     */
    public function pdf_output( $cert_id, $source_id, $user_id ) {
        header('Content-Disposition: attachment; filename="myfile.pdf"');
        header("Content-Type: application/pdf");

        ob_start();

        if ( has_action( 'learndash_tcpdf_init' ) ) {
            do_action(
                'learndash_tcpdf_init',
                array(
                    'cert_id' => $cert_id,
                    'user_id' => $user_id,
                    'post_id' => $source_id
                )
            );
        } else {
            require_once Helper::ld_pdf_converter_path();

            learndash_certificate_post_shortcode(
                array(
                    'cert_id' => $cert_id,
                    'user_id' => $user_id,
                    'post_id' => $source_id
                )
            );
        }

        echo ob_get_clean();
    }

    /**
     * Download PDF generated by LD.
     *
     * @param object $pdf TCPDF.
     * @param array $cert_args Certificate args.
     * 
     * PDF output mode:
     * 
     * I: send the file inline to the browser (default).
     * D: send to the browser and force a file download with the name given by name.
     * F: save to a local server file with the name given by name.
     * S: return the document as a string (name is ignored).
     * FI: equivalent to F + I option
     * FD: equivalent to F + D option
     * E: return the document as base64 mime multi-part email attachment (RFC 2045)
     * 
     * @return void
     */
    public function ld_download_pdf( $pdf, $cert_args ) {
        if ( URL::get_pdf_action() == URL::PDF_ACTION_DOWNLOAD ) {
            $pdf_file_name = $this->certificate->get_pdf_download_file_name();
            $pdf_file_name = ! empty ( $pdf_file_name ) ? $pdf_file_name : $cert_args['pdf_title'];

            $pdf->Output( $pdf_file_name . '.pdf', 'D' );
            exit;
        }
    }

    /**
     * Download PDF generated by LD Certificate Builder.
     * 
     * @param string $pdf_mode PDF mode.
     * 
     * I: send the file inline to the browser (default).
     * D: send to the browser and force a file download with the name given by name.
     * F: save to a local server file with the name given by name.
     * S: return the document as a string (name is ignored).
     * FI: equivalent to F + I option
     * FD: equivalent to F + D option
     * E: return the document as base64 mime multi-part email attachment (RFC 2045)
     *
     * 
     * @return string Changed PDF mode.
     */
    public function ld_certificate_builder_download_pdf( $pdf_mode ) {
        if ( URL::get_pdf_action() == URL::PDF_ACTION_DOWNLOAD ) {
            $pdf_mode = 'D';
        }

        return $pdf_mode;
    }

    /**
     * Change PDF file name generated by LD Certificate Builder.
     *
     * @param string $file_name PDF file name.
     * 
     * @return string Changed PDF file name.
     */
    public function ld_certificate_builder_download_file_name( $file_name ) {
        if ( URL::get_pdf_action() == URL::PDF_ACTION_DOWNLOAD ) {
            $pdf_file_name = $this->certificate->get_pdf_download_file_name();

            if ( ! empty ( $pdf_file_name ) ) {
                $file_name = $pdf_file_name;
            }
        }

        return $file_name;
    }

    /**
     * Output PDF image.
     *
     * @return void
     */
    public function pdf_image_output() {
        global $wp_query;

        if ( URL::get_pdf_action() == URL::PDF_ACTION_IMAGE ) {
            $image = $this->certificate->get_pdf_image_md5_path();

            if ( Helper::wp_filesystem()->exists( $image ) && mime_content_type( $image ) == self::PDF_IMAGE_MIME_TYPE ) {
                header( 'Content-type: ' . self::PDF_IMAGE_MIME_TYPE );
                readfile( $image );
                exit;
            }
        }

        $wp_query->set_404();
        status_header( 404 );
        get_template_part( 404 );
        exit;
    }

    /**
     * Render iframe_content to convert image from PDF.
     *
     * @return string HTML content of iframe.
     */
    public function render_pdf2image_iframe_content() {
        $ld_cvss_pdf2image_data = array();

        if ( $this->certificate ) {
            $ld_cvss_pdf2image_data = array(
                'ajax_url'          => admin_url( 'admin-ajax.php' ),
                'ajax_nonce'        => wp_create_nonce( self::AJAX_ACTION_PDF2IMAGE ),
                'ajax_action'       => self::AJAX_ACTION_PDF2IMAGE,
                'csuid'             => $this->certificate->get_csuid(),
                'worker'            => LD_CVSS_URL_ASSETS_JS . '/pdf-worker' . LD_CVSS_ASSETS_SUFFIX . '.js',
                'pdf'               => $this->certificate->get_pdf_view_url(),
                'mime_type'         => self::PDF_IMAGE_MIME_TYPE,
                'is_purchased'      => Certificate_Access::is_purchased( $this->certificate ) ? 'true' : 'false'
            );
        }

        return Template::render( 'pdf2image.php', compact( 'ld_cvss_pdf2image_data' ) );
    }

    /**
     * AJAX action to convert image from PDF.
     *
     * @return void
     */
    public function ajax_pdf2image() {
        if ( ! check_ajax_referer( self::AJAX_ACTION_PDF2IMAGE, 'nonce' ) ) {
            exit;
        }

        $csuid  = ! empty( $_POST['csuid'] ) ? strtoupper( $_POST['csuid'] ) : '';
        $image  = ! empty( $_POST['image'] ) ? $_POST['image'] : '';

        if ( ! $csuid || ! $image ) {
            exit;
        }

        $certificate_csuid  = Helper::decode_csuid( $csuid );
        $certificate        = new Certificate( $certificate_csuid['cert_id'], $certificate_csuid['source_id'], $certificate_csuid['user_id'] );

        if ( ! $certificate->is_valid() ) {
            exit;
        }

        $image_data = explode( ',', $image );
        $image_path = $certificate->get_pdf_image_md5_path();

        if ( ! empty( $image_data[1] ) && $image_path ) {
            Helper::wp_filesystem()->put_contents( $image_path, base64_decode( $image_data[1] ) );
        }
    }

    /**
     * AJAX action to handle search form.
     *
     * @return void
     */
    public function ajax_search_form() {
        if ( ! check_ajax_referer( self::AJAX_ACTION_SEARCH_FORM, 'nonce' ) ) {
            wp_send_json( array( 'message' => __( 'A security error has occurred. Please refresh the page and try again.', 'LD-CVSS' ) ), 403 );
        }

        $data = array(
            'certificate' => array()
        );

        $csuid = ! empty( $_POST['csuid'] ) ? strtoupper( $_POST['csuid'] ) : '';

        if ( $csuid ) {
            $certificate_csuid  = Helper::decode_csuid( $csuid );
            $certificate        = new Certificate( $certificate_csuid['cert_id'], $certificate_csuid['source_id'], $certificate_csuid['user_id'] );
    
            if ( $certificate->is_valid() && Certificate_Access::is_viewable( $certificate ) ) {
                $data['certificate'] = array(
                    'recipient'     => $certificate->get_recipient(),
                    'title'         => $certificate->get_title(),
                    'type'          => ucfirst( $certificate->get_type() ),
                    'date'          => date_i18n( apply_filters( 'ld_cvss_certificate_search_results_issue_date_format', $this->options->get( 'certificate_issue_date_format', get_option( 'date_format' ) ) ), $certificate->get_issue_date() ),
                    'public_url'    => $certificate->get_public_url()
                );
            }
        }

       $search_result = Template::get( 'search-form-results.php', $data );

       wp_send_json( compact( 'search_result' ) );
    }
}