<?php

namespace LD_CVSS\Classes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\Classes\Certificate_Access;

class Social_Buttons {

    /**
     * Get social networks.
     *
     * @param object $certificate Instance of Certificate class.
     * 
     * @return array Social networks.
     */
    private static function get_social_networks( $certificate ) {
        $social_networks = array(
            'twitter'           => array(
                'title' => 'Twitter',
                'url'   => 'https://twitter.com/intent/tweet?url=%url%&text=%text%'
            ),
            'facebook'          => array(
                'title' => 'Facebook',
                'url'   => 'https://www.facebook.com/sharer/sharer.php?u=%url%'
            ),
            'linkedin'          => array(
                'title' => 'LinkedIn',
                'url'   => 'https://www.linkedin.com/shareArticle?url=%url%'
            ),
            'linkedin_profile'  => array(
                'title' => '',
                'url'   => 'https://www.linkedin.com/profile/add?name=%name%&organizationId=%organization_id%&issueYear=%issue_year%&issueMonth=%issue_month%&certUrl=%cert_url%&certId=%cert_id%'
            )
        );

        $replacement = apply_filters( 'ld_cvss_social_buttons_replacement',
            array(
                '%url%'                 => urlencode( $certificate->get_public_url() ),
                '%text%'                => $certificate->get_title(),
                '%start_task%'          => 'CERTIFICATION_NAME',
                '%name%'                => $certificate->get_title(),
                '%organization_id%'     => Options::get_instance()->get( 'linkedin_company_id', 1337 ),
                '%issue_year%'          => date( 'Y', $certificate->get_issue_date() ),
                '%issue_month%'         => date( 'm', $certificate->get_issue_date() ),
                '%cert_url%'            => urlencode( $certificate->get_public_url() ),
                '%cert_id%'             => $certificate->get_id()
            ),
            $certificate
        );

        foreach ( $social_networks as $social_network_id => &$social_network ) {
            $social_network['url'] = str_replace( array_keys( $replacement ), array_values( $replacement ), $social_network['url'] );
        }

        return apply_filters( 'ld_cvss_social_networks', $social_networks, $certificate );
    }

    /**
     * Get social buttons.
     *
     * @param object $certificate Instance of Certificate class.
     * 
     * @return string HTML content of Social buttons.
     */
    public static function get( $certificate ) {
        if ( ! is_a( $certificate, 'LD_CVSS\\Classes\\Certificate' ) ) {
            return '';
        }

        if ( ! $certificate->is_valid() ) {
            return '';
        }

        if ( ! Certificate_Access::is_purchased( $certificate ) ) {
            return '';
        }

        $social_networks    = self::get_social_networks( $certificate );
        $iframe             = $certificate->get_pdf2image_iframe();

        return Template::get( 'social-buttons.php', compact( 'social_networks', 'iframe' ) );
    }

    /**
     * Render social buttons.
     *
     * @param object $certificate Instance of Certificate class.
     * 
     * @return void
     */
    public static function render( $certificate ) {
        echo self::get( $certificate );
    }
}