<?php

namespace LD_CVSS\Classes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

class Options {
    use \LD_CVSS\Traits\Trait_Singleton;

    /**
     * @var string Options key.
     */
    const OPTIONS_KEY = 'ld_cvss_options';

    /**
     * @var null|array Current options.
     */
    private $options = null;

    /**
     * Constructor.
     * 
     * @return void
     */
    protected function __construct() {
        $this->init();
    }

    /**
     * Init options.
     * 
     * @return void
     */
    private function init() {
        $this->options = apply_filters( 'ld_cvss_get_options', get_option( self::OPTIONS_KEY, null ) );

        if ( is_null( $this->options ) ) {
            $this->set_default();
        } else {
            $this->check_upgrade();
        }
    }

    /**
     * Get default options.
     * 
     * @return array Default options.
     */
    public function get_default() {
        return apply_filters( 'ld_cvss_get_default_options', array(
            'version'                           => LD_CVSS_VERSION,
            'verification_page_id'              => 0,
            'verification_page_template'        => 'template-01',
            'certificate_issue_date_format'     => get_option( 'date_format' ),
            'social_sharing_with_image'         => true,
            'linkedin_company_id'               => '1337',
            'license_key'                       => ''
        ) );
    }

    /**
     * Set default options.
     * 
     * @return void
     */
    public function set_default() {
        $this->set_all( $this->get_default() );
        $this->save();
    }

    /**
     * Check if options need upgrade.
     * 
     * @return object Instance of this class.
     */
    public function check_upgrade() {
        if ( defined( 'LD_CVSS_VERSION' ) && version_compare( LD_CVSS_VERSION, $this->get( 'version' ), '>' ) ) {
            $upgrade = array(
                'default_options'   => $this->get_default(),
                'options'           => $this->options
            );
            do_action_ref_array( 'ld_cvss_before_upgrade_options', array( &$upgrade ) );
            $this->options = wp_parse_args( $upgrade['options'], $upgrade['default_options'] );
            $this->save();
        }

        return $this;
    }

    /**
     * Get option by key.
     * 
     * @param string $key Option key.
     * 
     * @return object Instance of this class.
     */
    public function get( $key, $default = null ) {
        $option = array_key_exists( $key, $this->options ) ? $this->options[ $key ] : null;

        return is_null( $option ) && ! is_null( $default ) ? $default : $option;
    }

    /**
     * Get all options.
     * 
     * @return null|array Options.
     */
    public function get_all() {
        return $this->options;
    }

    /**
     * Set option by key.
     *
     * @param string $key Option key.
     * @param mixed $data Option value.
     * 
     * @return object Instance of this class.
     */
    public function set( $key, $data ) {
        if ( array_key_exists( $key, $this->options ) ) {
            $this->options[ $key ] = $data;
        }

        return $this;
    }

    /**
     * Set all options.
     *
     * @param null|array $data
     * 
     * @return object Instance of this class.
     */
    public function set_all( $data ) {
        $this->options = $data;

        return $this;
    }

    /**
     * Save options.
     * 
     * @return object Instance of this class.
     */
    public function save() {
        $this->options = apply_filters( 'ld_cvss_save_options', $this->options );
        update_option( self::OPTIONS_KEY, $this->options );

        return $this;
    }

    /**
     * Delete options.
     * 
     * @return object Instance of this class.
     */
    public function delete() {
        delete_option( self::OPTIONS_KEY );

        return $this;
    }
}