<?php

namespace LD_CVSS\Classes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

abstract class Notice {
    /**
     * @var array Notices.
     */
    protected $notices = array();

    /**
     * Get CSS class.
     *
     * @return string CSS class.
     */
    abstract public function get_css_class();

    /**
     * Get notice key by notice ID.
     * 
     * @param string $id Notice ID.
     *
     * @return int|boolean Notice key.
     */
    private final function get_key( $id ) {
        return array_search( $id, array_column( $this->notices, 'id' ) );
    }

    /**
     * Check if notice exists by notice ID.
     * 
     * @param string $id Notice ID.
     *
     * @return boolean Result of checking.
     */
    final public function exists( $id ) {
        return $this->get_key( $id ) !== false;
    }

    /**
     * Get count of notices.
     *
     * @return int Count of notices.
     */
    final public function get_count() {
        return count( $this->notices );
    }

    /**
     * Check if notices more than 0.
     *
     * @return boolean Result of checking.
     */
    final public function has_notices() {
        return ! empty( $this->notices );
    }

    /**
     * Add notice.
     *
     * @param string $id Notice ID.
     * @param string $text Notice text.
     * @param boolean $is_dismissible Is notice dismissible.
     * 
     * @return object Instance of this class.
     */
    final public function add( $id, $text, $is_dismissible = true ) {
        if ( ! empty( $id ) && ! $this->exists( $id ) ) {
            $this->notices[] = compact( 'id', 'text', 'is_dismissible' );
        }

        return $this;
    }

    /**
     * Change notice text.
     *
     * @param string $id Notice ID.
     * @param string $text Notice text.
     * 
     * @return object Instance of this class.
     */
    final public function change_text( $id, $text ) {
        if ( $this->exists( $id ) ) {
            $this->notices[ $this->get_key( $id ) ]['text'] = $text;
        }

        return $this;
    }

    /**
     * Change notice dismissible.
     *
     * @param string $id Notice ID.
     * @param boolean $is_dismissible Is notice dismissible.
     * 
     * @return object Instance of this class.
     */
    final public function change_dismissible( $id, $is_dismissible ) {
        if ( $this->exists( $id ) && is_bool( $is_dismissible ) ) {
            $this->notices[ $this->get_key( $id ) ]['is_dismissible'] = $is_dismissible;
        }

        return $this;
    }

    /**
     * Get notice by id.
     *
     * @param string $id Notice ID.
     * @param string $field (optional) Notice field.
     * 
     * @return null|string|array Notice|Notice field.
     */
    final public function get( $id, $field = '' ) {
        if ( ! $this->exists( $id ) ) {
            return null;
        }

        if ( $field ) {
            return isset( $this->notices[ $this->get_key( $id ) ][ $field ] ) ? $this->notices[ $this->get_key( $id ) ][ $field ] : '';
        }

        return $this->notices[ $this->get_key( $id ) ];
    }

    /**
     * Get all notices.
     *
     * @return array All notices.
     */
    final public function get_all() {
        return $this->notices;
    }

    /**
     * Remove notice by id.
     *
     * @param string $id Notice ID.
     * 
     * @return object Instance of this class.
     */
    final public function remove( $id ) {
        if ( $this->exists( $id ) ) {
            unset( $this->notices[ $this->get_key( $id ) ] );
        }

        return $this;
    }

    /**
     * Remove all notices.
     *
     * @return object Instance of this class.
     */
    final public function remove_all() {
        $this->notices = array();

        return $this;
    }

    /**
     * Get HTML of notice by id.
     * 
     * @param string $id Notice ID.
     *
     * @return string HTML of notice.
     */
    final public function get_html( $id ) {
        if ( ! $this->exists( $id ) ) {
            return '';
        }

        return sprintf( $this->get_html_template(), $id, $this->get_css_class(), ( $this->get( $id, 'is_dismissible' ) ? 'is-dismissible' : '' ), $this->get( $id, 'text' ) );
    }

    /**
     * Get HTML/array of HTML of all notices.
     * 
     * @param boolean $array Format of return.
     *
     * @return string|array HTML/array of HTML of all notices.
     */
    final public function get_all_html( $array = false ) {
        $notices = array();

        foreach ( $this->notices as $notice ) {
            extract( $notice );

            $notices[] = sprintf( $this->get_html_template(), $id, $this->get_css_class(), ( $is_dismissible ? 'is-dismissible' : '' ), $text );
        }

        return ! $array ? implode( '', $notices ) : $notices;
    }

    /**
     * Get HTML of notice template.
     *
     * @return string HTML of notice template.
     */
    final private function get_html_template() {
        return '<div id="ld-cvss-notice-%1$s" class="ld-cvss-notice notice notice-%2$s %3$s"><p>%4$s</p></div>';
    }
}