<?php

namespace LD_CVSS\Classes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

class Helper {

    /**
     * Init and get WP_Filesystem.
     *
     * @return object WP_Filesystem.
     */
    public static function wp_filesystem() {
        global $wp_filesystem;

        require_once ABSPATH . 'wp-admin/includes/file.php';

        if ( ! $wp_filesystem ) {
            WP_Filesystem();
        }

        return $wp_filesystem;
    }

    /**
     * Check if CSUID valid.
     *
     * @param string $csuid CSUID.
     * 
     * @return bool Result of checking.
     */
    public static function is_csuid_valid( $csuid ) {
        return preg_match( '/^[A-F0-9]+\-[A-F0-9]+\-[A-F0-9]+$/', $csuid );
    }

    /**
     * Encode data to CSUID.
     *
     * @param int $cert_id Certificate ID.
     * @param int $source_id Source ID.
     * @param int $user_id User ID.
     * 
     * @return string CSUID.
     */
    public static function encode_csuid( $cert_id, $source_id, $user_id ) {
        if ( ! $cert_id || ! $source_id || ! $user_id ) {
            return '';
        }

        $cid    = dechex( implode( '', array_map( 'ord', str_split( $cert_id + $user_id ) ) ) );
        $sid    = dechex( implode( '', array_map( 'ord', str_split( $source_id + $user_id ) ) ) );
        $uid    = dechex( implode( '', array_map( 'ord', str_split( $user_id ) ) ) );

        $csuid = sprintf( '%s-%s-%s', $cid, $sid, $uid );

        return strtoupper( $csuid );
    }

    /**
     * Decode CSUID to data.
     *
     * @param int $csuid CSUID.
     * 
     * @return array Data.
     */
    public static function decode_csuid( $csuid ) {
        if ( ! self::is_csuid_valid( $csuid ) ) {
            return array( 
                'cert_id'       => 0, 
                'source_id'     => 0, 
                'user_id'       => 0
            );
        }

        list( $cid, $sid, $uid ) = explode( '-', $csuid );

        $cert_id        = ( int ) implode( '', array_map( 'chr', str_split( hexdec( $cid ), 2 ) ) );
        $source_id      = ( int ) implode( '', array_map( 'chr', str_split( hexdec( $sid ), 2 ) ) );
        $user_id        = ( int ) implode( '', array_map( 'chr', str_split( hexdec( $uid ), 2 ) ) );

        $cert_id        = $cert_id - $user_id;
        $source_id      = $source_id - $user_id;

        return compact( 'cert_id', 'source_id', 'user_id' );
    }

    /**
     * Get path to LD PDF converter.
     *
     * @return void
     */
    public static function ld_pdf_converter_path() {
        return LEARNDASH_LMS_PLUGIN_DIR . '/includes/ld-convert-post-pdf.php';
    }

    /**
     * Check if LearDash plugin is active.
     * 
     * @return bool Result of checking.
     */
    public static function is_learndash_active() {
        require_once ABSPATH . '/wp-admin/includes/plugin.php';

        return is_plugin_active( 'sfwd-lms/sfwd_lms.php' );
    }

    public static function is_ld_pay_for_quiz_certificate_active() {
        require_once ABSPATH . '/wp-admin/includes/plugin.php';

        return is_plugin_active( 'learndash-pay-for-quiz-or-certificate/learndash-pay-for-quiz-certificate.php' );
    }
}