<?php

namespace LD_CVSS\Classes;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\URL;
use LD_CVSS\Frontend;
use LD_CVSS\Classes\Certificate_Sources\Certificate_Source_Course;
use LD_CVSS\Classes\Certificate_Sources\Certificate_Source_Quiz;
use LD_CVSS\Classes\Certificate_Sources\Certificate_Source_Group;
use LD_CVSS\Metaboxes\Certificate_Description;

class Certificate {
    /**
     * @var object Certificate WP_Post.
     */
    private $certificate;

    /**
     * @var object Source WP_Post.
     */
    private $source;

    /**
     * @var int User ID.
     */
    private $user_id;

    /**
     * @var object Verification page.
     */
    private $verification_page;

    /**
     * Constructor
     *
     * @param int $cert_id Certificate ID.
     * @param int $source_id Source ID.
     * @param int $user_id User ID.
     * 
     * @return void
     */
    public function __construct( $cert_id, $source_id, $user_id ) {
        $this->verification_page    = Verification_Page::get_instance();
        $certificate    = get_post( $cert_id );
        $source         = get_post( $source_id );

        $source_post_types  = array(
            learndash_get_post_type_slug( 'course' )    => Certificate_Source_Course::class,
            learndash_get_post_type_slug( 'quiz' )      => Certificate_Source_Quiz::class,
            learndash_get_post_type_slug( 'group' )     => Certificate_Source_Group::class,
        );

        if (
            $certificate && $certificate->post_type == learndash_get_post_type_slug( 'certificate' ) &&
            $source && array_key_exists( $source->post_type, $source_post_types )
        ) {
            $source_class       = $source_post_types[ $source->post_type ];
            $this->certificate  = $certificate;
            $this->source       = new $source_class( $source, $user_id );
            $this->user_id      = absint( $user_id );
        }
    }

    /**
     * Get certificate ID.
     *
     * @return int Certificate ID.
     */
    public function get_id() {
        return $this->certificate ? $this->certificate->ID : 0;
    }

    /**
     * Get certificate title.
     *
     * @return string Certificate title.
     */
    public function get_title() {
        return $this->get_id() ? $this->certificate->post_title : '';
    }

    /**
     * Get certificate description.
     *
     * @return string Certificate description.
     */
    public function get_description() {
        $description = '';

        if ( 
            ! empty( get_the_excerpt( $this->get_id() ) ) &&
            get_post_meta( $this->get_id(), Certificate_Description::META_KEY_MIGRATED , true ) != 'yes'
        ) {
            $description = get_the_excerpt( $this->get_id() );
        } else {
            $description = get_post_meta( $this->get_id(), Certificate_Description::META_KEY_DESCRIPTION, true );
        }

        return apply_filters( 'ld_cvss_certificate_description', $description );
    }

    /**
     * Get certificate permalink.
     *
     * @return string certificate permalink.
     */
    public function get_permalink() {
        return $this->get_id() ? get_the_permalink( $this->get_id() ) : '';
    }

    /**
     * Get post.
     *
     * @return object|null Certificate post (WP_Post).
     */
    public function get_post() {
        return $this->certificate;
    }

    /**
     * Get source.
     *
     * @return object|null Source post (WP_Post).
     */
    public function get_source() {
        return $this->source;
    }

    /**
     * Get source ID.
     *
     * @return int source ID.
     */
    public function get_source_id() {
        return $this->get_source() ? $this->get_source()->get_id() : 0;
    }

    /**
     * Get user ID.
     *
     * @return int user ID.
     */
    public function get_user_id() {
        return $this->user_id;
    }

    /**
     * Get certificate type.
     *
     * @return string certificate type.
     */
    public function get_type() {
        if ( $this->get_source() ) {
            return $this->get_source()->get_type();
        }

        return '';
    }

    /**
     * Check if certificate is valid.
     *
     * @return bool Result of checking.
     */
    public function is_valid() {
        if ( ! $this->get_id() ) {
            return false;
        }

        if ( ! $this->get_source() ) {
            return false;
        }

        if ( $this->get_id() != $this->get_source()->get_certificate_id() ) {
            return false;
        }

        if ( ! $this->get_recipient() ) {
            return false;
        }

        return $this->get_source()->is_completed();
    }

    /**
     * Get certificate course.
     *
     * @return int|WP_Post|array Certificate course.
     */
    public function get_course() {
        if ( $this->get_source() ) {
            return $this->get_source()->get_course();
        }

        return null;
    }

    /**
     * Get certificate instructor.
     *
     * @return null|WP_User Instructor.
     */
    public function get_instructor() {
        $source = $this->get_source();

        return $source ? get_user_by( 'id', $this->source->get_instructor_id() ) : null;
    }

    /**
     * Get certificate recipient.
     *
     * @return null|WP_User Recipient.
     */
    public function get_recipient() {
        return $this->get_user_id() ? get_user_by( 'id', $this->get_user_id() ) : null;
    }

    /**
     * Get certificate issue date.
     *
     * @return int Issue date timestamp.
     */
    public function get_issue_date() {
        $source = $this->get_source();

        return $source ? strtotime( wp_date( 'Y-m-d H:i:s', $source->get_issue_date() ) ) : 0;
    }

    /**
     * Get certificate CSUID.
     *
     * @return string CSUID.
     */
    public function get_csuid() {
        return $this->is_valid() ? Helper::encode_csuid( $this->get_id(), $this->get_source_id(), $this->get_user_id() ) : '';
    }

    /**
     * Build URL based on parameters.
     *
     * @return string Built URL.
     */
    private function build_url( $dest_url, $action = '' ) {
        $url        = '';
        $csuid      = Helper::encode_csuid( $this->get_id(), $this->get_source_id(), $this->get_user_id() );
        $url_args   = array();

        if ( ! $dest_url || ! $csuid ) {
            return $url;
        }

        if ( $action ) {
            $url_args[ URL::QUERY_VAR_PDF_ACTION ] = $action;
        }

        $url_args[ URL::QUERY_VAR_CSUID ] = $csuid;

        if ( get_option( 'permalink_structure' ) ) {
            $url = untrailingslashit( $dest_url ) . '/' . user_trailingslashit( implode( '/', array_values( $url_args ) ) );
        } else {
            $url = add_query_arg( $url_args, $dest_url );
        }

        return $url;
    }

    /**
     * Get certificate public URL.
     *
     * @return string Certificate public URL.
     */
    public function get_public_url() {
        return $this->is_valid() ? $this->build_url( $this->verification_page->get_page_url() ) : '';
    }

    /**
     * Get certificate PDF view URL.
     *
     * @return string Certificate PDF view URL.
     */
    public function get_pdf_view_url() {
        return $this->is_valid() ? $this->build_url( $this->get_permalink(), URL::PDF_ACTION_VIEW ) : '';
    }

    /**
     * Get certificate PDF download URL.
     *
     * @return string Certificate PDF download URL.
     */
    public function get_pdf_download_url() {
        return $this->is_valid() ? $this->build_url( $this->get_permalink(), URL::PDF_ACTION_DOWNLOAD ) : '';
    }

    /**
     * Get certificate PDF to image URL.
     *
     * @return string Certificate PDF to image URL.
     */
    public function get_pdf2image_url() {
        return $this->is_valid() ? $this->build_url( $this->get_permalink(), URL::PDF_ACTION_PDF2IMAGE ) : '';
    }

    /**
     * Get certificate PDF image URL.
     *
     * @return string Certificate PDF image URL.
     */
    public function get_pdf_image_url() {
        return $this->is_valid() ? $this->build_url( $this->get_permalink(), URL::PDF_ACTION_IMAGE ) : '';
    }

    /**
     * Get certificate PDF download file name.
     *
     * @return string Certificate PDF download file name.
     */
    public function get_pdf_download_file_name() {
        return $this->is_valid() ? sanitize_file_name( sprintf( '%s_%s_%s.pdf', $this->get_post()->post_title, $this->get_source()->get_post()->post_title, $this->get_recipient()->display_name ) ) : '';
    }

    /**
     * Get certificate PDF image MD5.
     *
     * @return string Certificate PDF image MD5.
     */
    public function get_pdf_image_md5() {
        return $this->is_valid() ? md5( $this->get_id() . $this->get_source_id() . $this->get_user_id() ) : '';
    }

    /**
     * Get certificate PDF image MD5 path.
     *
     * @return string Certificate PDF image MD5 path.
     */
    public function get_pdf_image_md5_path() {
        $image_extensions = array(
            'image/jpeg'    => 'jpg',
            'image/png'     => 'png'
        );

        return $this->is_valid() && array_key_exists( Frontend::PDF_IMAGE_MIME_TYPE, $image_extensions ) ? LD_CVSS_PATH_CERTIFICATES . '/' . $this->get_pdf_image_md5() . '.' . $image_extensions[ Frontend::PDF_IMAGE_MIME_TYPE ] : '';
    }

    /**
     * Get certificate PDF to image iframe.
     *
     * @return string Certificate PDF image MD5 path.
     */
    public function get_pdf2image_iframe() {
        if ( ! $this->is_valid() ) {
            return '';
        }

        return sprintf( 
            '<iframe src="%s" style="display: none;" width="1px" height="1px" frameborder="0"></iframe>',
            $this->get_pdf2image_url()
        );
    }

    /**
     * Get certificate ID.
     *
     * @return array certificate ID.
     */
    public function get_pdf_meta() {
        $meta = ( array ) get_post_meta( $this->get_id(), 'learndash_certificate_options', true );

        return wp_parse_args( $meta, array(
            'pdf_page_format'       => 'LETTER',
            'pdf_page_orientation'  => 'L'
        ) );
    }
}