<?php

namespace LD_CVSS;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\Classes\Certificate;
use LD_CVSS\Classes\Social_Buttons;
use LD_CVSS\Classes\Verification_Page;

class Certificate_Hooks {
    use \LD_CVSS\Traits\Trait_Singleton;

    /**
     * @var string JSON param to display social buttons in Quizzes.
     */
    const JSON_PARAM = 'ld_cvss_social_buttons';
    const CERTIFICATE_URL_KEY = 'ld_cvss_quiz_certificate_url_key';

    private $verification_page;

    /**
     * Constructor.
     * 
     * @return void
     */
    public function __construct() {
        $this->verification_page = Verification_Page::get_instance();

        add_action( 'wp_enqueue_scripts', array( $this, 'quiz_assets' ) ) ;
        add_filter( 'learndash_course_certificate_link', array( $this, 'certificate_course_group_link' ), 11, 3 );
        add_filter( 'learndash_group_certificate_link', array( $this, 'certificate_course_group_link' ), 11, 3 );
        add_filter( 'learndash_certificate_details_link', array( $this, 'certificate_quiz_link' ), 11, 4 );
        add_action( 'learndash-alert-between-message-button', array( $this, 'certificate_course_group_message' ), 11, 4 );
        add_filter( 'ld_after_course_status_template_container', array( $this, 'certificate_legacy_course_message' ), 11, 4 );
        add_filter( 'learndash_quiz_completed_result_settings', array( $this, 'certificate_quiz_completion_json' ), 11, 2 );
        add_filter( 'ld_certificate_link_label', array( $this, 'certificate_quiz_button_label' ), 11, 3 );
        add_action( 'learndash-quiz-row-title-after', array( $this, 'certificate_icon_in_course_quiz_list'), 11, 3 );
    }

    /**
     * Enqueue quiz assets.
     * 
     * @return void
     */
    public function quiz_assets() {
        $post = get_post();

        if ( $post && $post->post_type == learndash_get_post_type_slug( 'quiz' ) ) {
            wp_enqueue_script( 'ld_cvss-quiz-social-buttons', LD_CVSS_URL_ASSETS_JS . '/quiz-social-buttons' . LD_CVSS_ASSETS_SUFFIX . '.js', array( 'jquery' ), true, LD_CVSS_VERSION );
            wp_localize_script( 'ld_cvss-quiz-social-buttons', 'ld_cvss_quiz_data', array(
                'json_param' => self::JSON_PARAM,
                'certificate_url' => self::CERTIFICATE_URL_KEY,
            ) );
        }
    }

    /**
     * Replace standard certificate link for courses and groups with plugin link.
     *
     * @param string $url Course certificate link.
     * @param int $source_id Course/Group ID.
     * @param int $user_id User ID.
     * 
     * @return string Replaced link.
     */
    public function certificate_course_group_link( $url, $source_id, $user_id ) {
        if ( $this->verification_page->page_exists() ) {
            $cert_id = ( int ) learndash_get_setting( $source_id, 'certificate' );

            if ( $cert_id ) {
                $certificate = new Certificate( $cert_id, $source_id, $user_id );

                if ( $certificate->is_valid() ) {
                    $url = $certificate->get_public_url();
                }
            }
        }

        return $url;
    }

    /**
     * Replace standard quiz certificate link with plugin link.
     *
     * @param string $url Quiz certificate link.
     * @param $certificate_post WP_Post of certificate.
     * @param int $quiz_id Quiz ID.
     * @param int $user_id User ID.
     * 
     * @return string Replaced link.
     */
    public function certificate_quiz_link( $url, $certificate_post, $quiz_id, $user_id  ) {
        if ( $this->verification_page->page_exists() ) {
            $cert_id = ( int ) learndash_get_setting( $quiz_id, 'certificate' );

            if ( $cert_id ) {
                $certificate = new Certificate( $cert_id, $quiz_id, $user_id );

                if ( $certificate->is_valid() ) {
                    $url = $certificate->get_public_url();
                }
            }
        }

        return $url;
    }

    /**
     * Add social buttons and certificate button to Course/Group alert messages.
     *
     * @param string $class List of alert CSS classes.
     * @param string $icon List of alert icon CSS classes.
     * @param string $message Alert message text.
     * @param string $type Alert message type.
     * 
     * @return string Replaced link.
     */
    public function certificate_course_group_message( $class, $icon, $message, $type ) {
        if ( 'success ld-alert-certificate' != $type ) {
            return;
        }

        $source = get_post();

        if ( $source && in_array( $source->post_type, array( learndash_get_post_type_slug( 'course' ), learndash_get_post_type_slug( 'group' ) ) ) ) {
            $user_id    = get_current_user_id();
            $cert_id    = ( int ) learndash_get_setting( $source->ID, 'certificate' );

            if ( $cert_id ) {
                $certificate = new Certificate( $cert_id, $source->ID, $user_id );
    
                if ( $certificate->is_valid() ) {
                    if ( $this->verification_page->page_exists() ) {
                        Social_Buttons::render( $certificate );
                    }

                    add_filter( 'ld-alert-button', array( $this, 'certificate_downoad_button' ) );
                }
            }
        }
    }

    /**
     * Change css class and label for certificate button.
     *
     * @param array $args Button args.
     * 
     * @return array Changed button args.
     */
    public function certificate_downoad_button( $args ) {
        $args['class']  = 'ld-cvss-button ld-cvss-button-download-certificate';
        $args['label']  = __( 'Certificate', 'LD-CVSS' );

        return $args;
    }

    /**
     * Add social buttons to course alert messages in Legacy theme.
     *
     * @param string $class List of alert CSS classes.
     * @param string $icon List of alert icon CSS classes.
     * @param string $message Alert message text.
     * @param string $type Alert message type.
     * 
     * @return string Replaced link.
     */
    public function certificate_legacy_course_message( $content, $course_status, $course_id, $user_id ) {
        if ( \LearnDash_Theme_Register::get_active_theme_key() == 'legacy' ) {
            $cert_id = ( int ) learndash_get_setting( $course_id, 'certificate' );

            if ( $cert_id ) {
                $certificate = new Certificate( $cert_id, $course_id, $user_id );
    
                if ( $certificate->is_valid() ) {
                    if ( $this->verification_page->page_exists() ) {
                        $content = Social_Buttons::get( $certificate );
                    }
                }
            }
        }

        return $content;
    }

    /**
     * Display social buttons in quiz result.
     *
     * @param array $quiz_result_settings Quiz result settings.
     * @param array $quiz_data Quiz data.
     * 
     * @return array Changed quiz result settings.
     */
    public function certificate_quiz_completion_json( $quiz_result_settings, $quiz_data ) {
        $user_id    = get_current_user_id();
        $quiz_id    = $quiz_data['quiz'];
        $cert_id    = ( int ) learndash_get_setting( $quiz_id, 'certificate' );

        if ( $cert_id ) {
            $certificate = new Certificate( $cert_id, $quiz_id, $user_id );

            if ( $certificate->is_valid() ) {
                if ( $this->verification_page->page_exists() ) {
                    $certificate_details    = learndash_certificate_details( $quiz_id, $user_id );
                    $certificate_url        = ! empty( $certificate_details['certificateLink'] ) ? $certificate_details['certificateLink'] : '';
                    if ( $certificate_url ) {
                        $quiz_result_settings[ self::JSON_PARAM ] = Social_Buttons::get( $certificate );
                        $quiz_result_settings[ self::CERTIFICATE_URL_KEY ] = $certificate_url;
                    }
                }
            }
        }

        return $quiz_result_settings;
    }

    /**
     * Change quiz certificate button label.
     *
     * @param string $label Button label.
     * @param int $user_id User ID.
     * @param int $quiz_id Quiz ID.
     * 
     * @return string Changed button label.
     */
    public function certificate_quiz_button_label( $label, $user_id, $quiz_id ) {
        return '<span class="ld-icon ld-icon-download"></span>' . __( 'Certificate', 'LD-CVSS' );
    }

    /**
     * Display certificate icon in Course quiz list.
     *
     * @param int $quiz_id Quiz ID.
     * @param int $course_id Course ID.
     * @param int $user_id User ID.
     * 
     * @return void
     */
    public function certificate_icon_in_course_quiz_list( $quiz_id, $course_id, $user_id ) {
        if ( get_post_type() != learndash_get_post_type_slug( 'course' ) ) {
            return;
        }

        $cert_id = ( int ) learndash_get_setting( $quiz_id, 'certificate' );

        if ( $cert_id ) {
            $certificate = new Certificate( $cert_id, $quiz_id, $user_id );

            if ( $certificate->is_valid() ) {
                $social_buttons         = '';
                $certificate_details    = learndash_certificate_details( $quiz_id, $user_id );
                $certificate_url        = ! empty( $certificate_details['certificateLink'] ) ? $certificate_details['certificateLink'] : '';

                if ( $this->verification_page->page_exists() ) {
                    $social_buttons = Social_Buttons::get( $certificate );
                    $certificate_url = $certificate->get_public_url();
                }

                if ( $certificate_url ) {
                    printf( '</a>%s<a class="btn-blue ld-cvss-btn-blue" href="%s" target="_blank"><span class="ld-icon ld-icon-download"></span>%s</a><a href="#" style="display: none;">', $social_buttons, $certificate_url, __( 'Certificate', 'LD-CVSS' ) );
                }
            }
        }
    }
}