<?php

namespace LD_CVSS;

/* Exit if accessed directly. */
defined( 'ABSPATH' ) || exit;

use LD_CVSS\License\License_Manager;
use LD_CVSS\Classes\Options;
use LD_CVSS\Classes\Notice_Manager;
use LD_CVSS\Classes\Template;

class Admin {
    use \LD_CVSS\Traits\Trait_Singleton;

    const PAGE_SLUG = 'ld-cvss-admin-dashboard';

    /**
     * @var string Settings form nonce action.
     */
    const FORM_NONCE_ACTION = 'ld_cvss_submit_settings_form';

    /**
     * @var string Settings form nonce name.
     */
    const FORM_NONCE_NAME = 'ld_cvss_nonce_settings_form';

    /**
     * @var string Settings form action save.
     */
    const FORM_ACTION_SAVE_SETTINGS = 'save_settings';

    /**
     * @var string Settings form action activate license key.
     */
    const FORM_ACTION_ACTIVATE_LICENSE_KEY = 'activate_license_key';

    /**
     * @var string Settings form action deactivate license key.
     */
    const FORM_ACTION_DEACTIVATE_LICENSE_KEY = 'deactivate_license_key';

    /**
     * @var string Settings form action deactivate license key.
     */
    const FORM_ACTION_CHECK_LICENSE_KEY = 'check_license_key';

    /**
     * @var object Plugin options.
     */
    private $options;

    /**
     * @var object License manager.
     */
    private $license_manager;

    /**
     * @var object Notice manager.
     */
    private $notice_manager;

    /**
     * Constructor.
     * 
     * @return void
     */
    public function __construct() {
        Metabox::get_instance();

        $this->options              = Options::get_instance();
        $this->license_manager      = License_Manager::get_instance();
        $this->notice_manager       = new Notice_Manager;

        add_filter( 'display_post_states', array( $this, 'verification_page_status' ), 10, 2 );
        add_filter( 'plugin_action_links', array( $this, 'plugin_action_links' ), 10, 4 );
        add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta_links' ), 10, 4 );
        add_action( 'admin_menu', array( $this, 'admin_menu' ), 11 );
    }

    public function verification_page_status( $post_states, $post ) {
        global $pagenow;

        if ( is_admin() && $post->post_type == 'page' && $pagenow == 'edit.php' ) {
            if ( $post->ID == $this->options->get( 'verification_page_id' ) ) {
                $post_states[] = sprintf( '%s <img src="%s" style="width:40px;height:34px;vertical-align: middle;">', __( 'Certificates Verification Page', 'LD_CVSS' ), LD_CVSS_URL_ASSETS_IMAGES . '/icon-state-verification-page.png' );
            }
        }

        return $post_states;
    }

    /**
     * Add an item to the plugin action links.
     * 
     * @param array $actions Current plugin actions.
     * @return string $plugin_file Current plugin file.
     * @return array $plugin_data Current plugin data.
     * @return string $context Current plugins.php page tab.
     * 
     * @return array $actions Modified plugin actions.
     */
    public function plugin_action_links( $actions, $plugin_file, $plugin_data, $context ) {
        if ( plugin_basename( LD_CVSS_FILE ) == $plugin_file ) {
            $settings_link = sprintf( '<a href="%s">%s</a>', esc_url( add_query_arg( 'page', self::PAGE_SLUG, admin_url( 'admin.php' ) ) ), __( 'Settings', 'LD-CVSS' ) );
            array_unshift( $actions, $settings_link );
        }

        return $actions;
    }

    public function plugin_row_meta_links( $plugin_meta, $plugin_file, $plugin_data, $status ) {
        if ( LD_CVSS_BASE == $plugin_file ) {
            $plugin_meta[] = sprintf( '<a href="https://docs.wooninjas.com/article/167-learndash-certificate-verify-share-add-on-overview" target="_blank">%s</a>', __( 'Documentation', 'LD-CVSS' ) );
            $plugin_meta[] = sprintf( '<a href="https://wooninjas.com/open-support-ticket/" target="_blank">%s</a>', __( 'Support', 'LD-CVSS' ) );
        }
        return $plugin_meta;
    }

    /**
     * Add an item to LearnDash admin menu.
     * 
     * @return void
     */
    public function admin_menu() {
        $admin_page = add_submenu_page( 'learndash-lms', 'گواهینامه لرن دش', 'گواهینامه لرن دش', 'manage_options', self::PAGE_SLUG, array( $this, 'render_admin_page' ) );
    }

    /**
     * Save settings.
     * 
     * @return void
     */
    public function save_settings( $data ) {
        if ( isset( $data['verification_page_id'] ) ) {
            $this->options->set( 'verification_page_id', ( int ) $data['verification_page_id'] );
        }

        if ( isset( $data['verification_page_template'] ) ) {
            $this->options->set( 'verification_page_template', $data['verification_page_template'] );
        }

        if ( isset( $data['certificate_issue_date_format'] ) ) {
            $this->options->set( 'certificate_issue_date_format', $data['certificate_issue_date_format'] );
        }

        if ( isset( $data['social_sharing_with_image'] ) ) {
            $this->options->set( 'social_sharing_with_image', $data['social_sharing_with_image'] == 'true' );
        }

        if ( isset( $data['linkedin_company_id'] ) ) {
            $this->options->set( 'linkedin_company_id', sanitize_text_field( $data['linkedin_company_id'] ) );
        }

        $this->options->save();
        
        $this->notice_manager->success->add( 'settings_saved', sprintf( '<strong>%s</strong>', __( 'Settings saved.', 'LD-CVSS' ) ) );
    }

    /**
     * Activate license key.
     * 
     * @return void
     */
    public function activate_license_key( $data ) {
        $this->license_manager->set_license_key( $data['license_key'] );
        $attempt_activate = $this->license_manager->activate();

        if ( is_wp_error( $attempt_activate ) ) {
            $this->notice_manager->error->add( 'activate_license_key', sprintf( '<strong>%s</strong>', $attempt_activate->get_error_message() ) );
        } else {
            $this->options->set( 'license_key', $this->license_manager->get_license_key() );
            $this->options->save();
            $this->notice_manager->success->add( 'activate_license_key', sprintf( '<strong>%s</strong>', __( 'License key has been activated.', 'LD-CVSS' ) ) );
        }
    }

    /**
     * Deactivate license key.
     * 
     * @return void
     */
    public function deactivate_license_key( $data ) {
        $this->license_manager->set_license_key( $data['license_key'] );
        $attempt_deactivate = $this->license_manager->deactivate();

        if ( is_wp_error( $attempt_deactivate ) ) {
            $this->notice_manager->error->add( 'deactivate_license_key', sprintf( '<strong>%s</strong>', $attempt_deactivate->get_error_message() ) );
        } else {
            $this->options->set( 'license_key', '' );
            $this->options->save();
            $this->notice_manager->success->add( 'deactivate_license_key', sprintf( '<strong>%s</strong>', __( 'License key has been deactivated.', 'LD-CVSS' ) ) );
        }
    }

    /**
     * Check license key.
     * 
     * @return void
     */
    public function check_license_key( $data ) {
        $this->license_manager->set_license_key( $data['license_key'] );
        $attempt_check = $this->license_manager->check( true );

        if ( is_wp_error( $attempt_check ) ) {
            $this->notice_manager->error->add( 'check_license_key', sprintf( '<strong>%s</strong>', $attempt_check->get_error_message() ) );
        } else {
            $this->notice_manager->success->add( 'check_license_key', sprintf( '<strong>%s</strong>', $this->license_manager->get_status() ) );
        }
    }

    /**
     * Render settings page.
     * 
     * @return void
     */
    public function render_admin_page() {
        if ( ! empty( $_POST['ld_cvss_submit_form'] ) && check_admin_referer( self::FORM_NONCE_ACTION, self::FORM_NONCE_NAME ) ) {
            if ( $_POST['ld_cvss_submit_form'] == self::FORM_ACTION_SAVE_SETTINGS ) {
                $this->save_settings( $_POST['ld_cvss_settings'] );
            } elseif ( $_POST['ld_cvss_submit_form'] == self::FORM_ACTION_ACTIVATE_LICENSE_KEY ) {
                $this->activate_license_key( $_POST['ld_cvss_license'] );
            } elseif ( $_POST['ld_cvss_submit_form'] == self::FORM_ACTION_DEACTIVATE_LICENSE_KEY ) {
                $this->deactivate_license_key( $_POST['ld_cvss_license'] );
            } elseif ( $_POST['ld_cvss_submit_form'] == self::FORM_ACTION_CHECK_LICENSE_KEY ) {
                $this->check_license_key( $_POST['ld_cvss_license'] );
            }
        }

        $page_url   = add_query_arg( array( 'page' => self::PAGE_SLUG ), admin_url( 'admin.php' ) );
        $tabs       = array(
            'general' => array(
                'title' => __( 'General', 'LD-CVSS' ),
                'url'   => $page_url
            ),
            
        );

        $active_tab         = ! empty( $_GET['tab'] ) && array_key_exists( $_GET['tab'], $tabs ) ? $_GET['tab'] : 'general';
        $current_page_url   = ! empty( $_GET['tab'] ) && array_key_exists( $_GET['tab'], $tabs ) ? add_query_arg( array( 'tab' => 'license' ), $page_url ) : $page_url;

        $shortcodes = array(
            'verification_page'         => Frontend::SHORTCODE_VERIFICATION_PAGE,
            'certificate_id'            => Frontend::SHORTCODE_CERTIFICATE_ID,
            'certificate_qr_code'       => Frontend::SHORTCODE_CERTIFICATE_QR_CODE
        );

        $page_ids           = get_posts( array(
            'post_type'     => 'page',
            'status'        => 'publish',
            'numberposts'   => -1,
            'fields'        => 'ids',
            'orderby'       => 'title',
            'order'         => 'ASC'
        ) );

        $pages              = array_combine( $page_ids, array_map( 'get_the_title', $page_ids ) );
        $page_templates     = array_combine( 
            Frontend::PAGE_TEMPLATES, 
            array(
                __( 'Template 01', 'LD-CVSS' ),
                __( 'Template 02', 'LD-CVSS' )
            )
        );

        $options            = $this->options;
        $license_manager    = $this->license_manager;
        $notice_manager     = $this->notice_manager;

        $is_license_activated = false;
        $license_key = $this->options->get( 'license_key', '' );

        if ( $license_key ) {
            $license_manager->set_license_key( $license_key );
            $attempt_check = $license_manager->check( true );
            $is_license_activated = ! is_wp_error( $attempt_check );
        }

        $form_actions = array(
            'save_settings'             => self::FORM_ACTION_SAVE_SETTINGS,
            'activate_license_key'      => self::FORM_ACTION_ACTIVATE_LICENSE_KEY,
            'deactivate_license_key'    => self::FORM_ACTION_DEACTIVATE_LICENSE_KEY,
            'check_license_key'         => self::FORM_ACTION_CHECK_LICENSE_KEY
        );

        $nonce_field = wp_nonce_field( self::FORM_NONCE_ACTION, self::FORM_NONCE_NAME, true, false );

        Template::render( 'admin/settings-page.php', array(
            'notice_manager'            => $notice_manager,
            'options'                   => $options,
            'page_url'                  => $page_url,
            'active_tab'                => $active_tab,
            'current_page_url'          => $current_page_url,
            'tabs'                      => $tabs,
            'shortcodes'                => $shortcodes,
            'pages'                     => $pages,
            'page_templates'            => $page_templates,
            'is_license_activated'      => $is_license_activated,
            'form_actions'              => $form_actions,
            'nonce_field'               => $nonce_field,
        ) );
    }
}