<?php

namespace Digits\SocialLogin\Auths;

use Google\Service\Oauth2;
use Google_Client;

class GoogleAuth extends SocialLoginAuth
{
    private $client_id;
    private $settings;
    private $redirect_url;

    function __construct($settings, $social_info, $icon_only)
    {
        $this->settings = $settings;
        $this->client_id = $settings['clientID'];
        $this->redirect_url = $social_info['redirect_uri'];

        parent::__construct();
    }

    public function use_js()
    {
        return !empty($this->settings['oneTapSignIn']);
    }

    public function render()
    {
        ?>
        <div>
            <?php
            if ($this->use_js()) {
                ?>
                <div id="g_id_onload"
                     data-client_id="<?php echo esc_attr($this->client_id); ?>"
                     data-context="use"
                     data-login_uri="<?php echo esc_attr($this->get_nonce_redirect_url()); ?>"
                     data-itp_support="true">
                </div>
                <?php
            }
            $this->default_layout('digits_google_login', 'Google');
            ?>
        </div>
        <?php
    }

    public function get_nonce_redirect_url()
    {
        return add_query_arg([
            'state' => wp_create_nonce('google_login')
        ], $this->redirect_url);
    }

    public function get_client()
    {
        $client = new Google_Client();
        $client->setApplicationName($this->settings['appName']);
        $client->setClientId($this->client_id);
        $client->setClientSecret($this->settings['clientSecretID']);
        $client->setRedirectUri($this->redirect_url);
        $client->setState(wp_create_nonce('google_login'));
        $client->setScopes(['https://www.googleapis.com/auth/userinfo.email',
            'https://www.googleapis.com/auth/userinfo.profile']);

        return $client;
    }

    public function get_url()
    {
        return $this->get_client()->createAuthUrl();
    }

    public function enqueue_scripts()
    {
        wp_enqueue_script('google-oauth', 'https://accounts.google.com/gsi/client');
    }

    public function process_token()
    {
    }

    public function process_oauth_token()
    {
        if (!wp_verify_nonce($_REQUEST['state'], 'google_login')) {
            $this->nonce_fail();
            die();
        }

        if (isset($_REQUEST['credential'])) {
            $this->verify_id_token();
        }
        if (empty($_REQUEST['code'])) {
            $this->invalid_info();
            die();
        }

        try {
            $code = $_REQUEST['code'];

            $client = $this->get_client();
            $client->fetchAccessTokenWithAuthCode($code);
            $access_token = $client->getAccessToken();

            if (empty($access_token)) {
                $this->invalid_info();
                die();
            }

            $oauth = new Oauth2($client);
            $user = $oauth->userinfo->get();

            $name = $user['name'];
            $username = $user['login'];
            $id = $user['id'];
            $email = '';
            if ($user['verifiedEmail'] == 1) {
                $email = $user['email'];
            }
            $response = digits_social_create_user($name, $email, 'google', $username, $id);

            $this->success_view($response);

        } catch (\Exception $e) {
            $this->error_view($e);
        }
    }

    public function verify_id_token()
    {
        try {
            $code = $_REQUEST['credential'];
            $client = $this->get_client();
            $token = $client->verifyIdToken($code);
            if ($token['aud'] != $this->client_id) {
                $this->invalid_info();
                die();
            }

            if($token['exp'] > time()){
                $this->invalid_info();
                die();
            }
            $name = $token['name'];
            $username = $token['login'];
            $id = $token['id'];
            $email = '';
            if ($token['email_verified'] == 1) {
                $email = $token['email'];
            }
            $response = digits_social_create_user($name, $email, 'google', $username, $id);

            $this->redirect($response);


        } catch (\Exception $e) {
            $this->error_view($e);
        }
    }
}
